use crate::helpers;
use crate::project;
use crate::runtime;

use colored::Colorize;
use esbuild_rs::{build as esBuild, BuildOptionsBuilder, Format, Platform};
use macros::{crashln, error, ternary};
use open::that;
use question::{Answer, Question};
use rustyline::{error::ReadlineError, Editor};
use shell::cmd;
use std::collections::HashMap;
use std::env;
use std::net::SocketAddr;
use std::time::Instant;

pub fn serve(host: String, port: i32, path: &String) {
    let addr: SocketAddr = format!("{host}:{}", port).parse().expect("Invalid server address");
    let runtime = tokio::runtime::Builder::new_current_thread().enable_all().build().unwrap();

    println!("{} {}", "listening on".bright_blue(), format!("{:?}", addr).cyan());
    println!("{} {}", "serving path".yellow(), format!("{path}").bright_yellow());

    that(format!("http://{:?}", addr)).unwrap_or_else(|e| eprintln!("Failed to open your default browser: {}", e));
    runtime.block_on(warp::serve(warp::fs::dir(path.clone())).run(addr));
}

pub fn setup() {
    log::info!("setup has been ran");
    let home_dir = home::home_dir().unwrap();
    let package_dir = format!("{}/.just/packages", home_dir.display());

    if !helpers::Exists::folder(package_dir.clone()).unwrap() {
        std::fs::create_dir_all(package_dir.clone()).unwrap();
        println!("created {}/.just/packages", &home_dir.display());
    }
}

pub fn build(entry: &String, path: &String) {
    let mut options_builder = BuildOptionsBuilder::new();
    let package = project::package::read();

    options_builder.entry_points.push(entry.clone());
    options_builder.keep_names = true;
    options_builder.minify_syntax = true;
    options_builder.minify_whitespace = true;
    options_builder.minify_identifiers = true;
    options_builder.splitting = true;
    options_builder.bundle = true;
    options_builder.write = true;
    options_builder.outdir = path.clone();
    options_builder.platform = Platform::Browser;
    options_builder.format = Format::ESModule;

    options_builder.external = vec![
        "just/io".to_string(),
        "just/sys".to_string(),
        "just/net".to_string(),
        "just/crypto".to_string(),
        "just/crypto:enc".to_string(),
        "just/db:kv".to_string(),
        "just/db:sqlite".to_string(),
    ];

    options_builder.banner = HashMap::from([(
        "js".to_string(),
        format!("/* {}@{} - This file is automatically generated. DO NOT EDIT */", package.info.name, package.info.version),
    )]);

    let options = options_builder.build();
    let runtime = tokio::runtime::Builder::new_current_thread().enable_all().build().unwrap();
    let start = Instant::now();

    log::debug!("entry point: {entry}");
    log::debug!("bundle dir: {path}");
    runtime.block_on(esBuild(options));
    log::info!("took {:.2?}", start.elapsed());
}

pub fn cache_clean() {
    let home_dir = home::home_dir().unwrap();
    let package_dir = format!("{}/.just/packages", home_dir.display());

    if helpers::Exists::folder(package_dir.clone()).unwrap() {
        std::fs::remove_dir_all(package_dir.clone()).unwrap();
        std::fs::create_dir_all(package_dir.clone()).unwrap();
        println!("removed cached packages");
    }
}

pub fn get_version(short: bool) -> String {
    return match short {
        true => format!("{} {}", env!("CARGO_PKG_NAME"), env!("CARGO_PKG_VERSION")),
        false => format!("{} ({} {})", env!("CARGO_PKG_VERSION"), env!("GIT_HASH"), env!("BUILD_DATE")),
    };
}

pub fn project_meta() {
    let package = project::package::read();
    println!("{} {}@{}", "starting".green(), format!("{}", package.info.name).bold(), format!("{}", package.info.version).bold());
}

pub fn run_task(task: &str) {
    let tasks = project::package::read().tasks;
    if tasks.get(task).is_none() {
        crashln!("The task '{task}' could not be found. Does it exist?");
    }

    log::debug!("{:?}", tasks.get(task));
    println!("\n{} task {}", "running".green(), task.bold());
    println!("{} {}\n", "»".white(), tasks[task]);

    for command in &tasks[task].split("&&").collect::<Vec<&str>>() {
        if let Err(error) = cmd!(command.trim()).run() {
            error!("{:?}", error);
        }
    }
}

pub fn run_test(test: &str) {
    let tests = project::package::read().tests;
    if tests.get(test).is_none() {
        crashln!("The test '{test}' could not be found. Does it exist?");
    }

    log::debug!("{:?}", tests.get(test));
    println!("\n{} test {}", "running".green(), test.bold());
    println!("{} {}\n", "»".white(), tests[test]);

    for command in &tests[test].split("&&").collect::<Vec<&str>>() {
        if let Err(error) = cmd!(command.trim()).run() {
            error!("{:?}", error);
        }
    }
}

pub fn list_tasks() {
    let tasks = project::package::read().tasks;
    project::tasks::task_list(tasks);
}

pub fn list_tests() {
    let tests = project::package::read().tests;
    project::tests::test_list(tests);
}

pub fn create_project_yml() {
    let exists: bool = std::path::Path::new("package.yml").exists();
    if !exists {
        project::init::create_project();
    } else {
        let answer = Question::new("overwrite project.yml?").show_defaults().confirm();

        ternary!(answer == Answer::YES, project::init::create_project(), println!("{}", "Aborting...".white()))
    }
}

pub fn run_exec(path: &str, silent: bool, code: &str) {
    let exists: bool = std::path::Path::new("package.yml").exists();
    let runtime = tokio::runtime::Builder::new_current_thread().enable_all().build().unwrap();

    if silent {
        if let Err(error) = runtime.block_on(runtime::exec(&path.to_string(), code.to_string())) {
            eprintln!("{}", format!("{}", error).red());
        }
    } else {
        ternary!(exists, project_meta(), {});
        let start = Instant::now();
        if let Err(error) = runtime.block_on(runtime::exec(&path.to_string(), code.to_string())) {
            eprintln!("{}", format!("{}", error).red());
        } else {
            let path = path.split("/").collect::<Vec<_>>();

            println!("\n{} took {}", format!("{}", path[path.len() - 1]).white(), format!("{:.2?}", start.elapsed()).yellow())
        }
    }
}

pub fn run_repl() {
    let runtime = tokio::runtime::Builder::new_current_thread().enable_all().build().unwrap();
    let mut readline_editor = Editor::<()>::new();
    let mut exit_value = 0;

    println!("{}", get_version(true));
    println!("Type \".help\" for more information.");

    loop {
        let readline = readline_editor.readline("> ");
        match readline {
            Ok(line) => {
                if line == ".help" {
                    println!(".clear    Clear the screen\n.exit     Exit the REPL\n.help     Print this help message")
                } else if line == ".clear" {
                    print!("{}[2J", 27 as char);
                } else if line == ".exit" {
                    break;
                } else {
                    if let Err(error) = runtime.block_on(runtime::repl(&line)) {
                        eprintln!("{}", format!("{}", error).red());
                    }
                }
            }
            Err(ReadlineError::Interrupted) => {
                exit_value += 1;
                if exit_value == 2 {
                    break;
                } else {
                    println!("(To exit, press Ctrl+C again, Ctrl+D or type .exit)");
                }
            }
            Err(ReadlineError::Eof) => {
                break;
            }
            Err(err) => {
                println!("Error: {:?}", err);
                break;
            }
        }
    }
}
